import copy

"""
cube state defined as 6 lists of 9 elements each
supported rotations are:
    
U,D,F,B,L,R (up,down,front,back,left,right)
1 (90 degrees clockwise)
2 (180 degrees clockwise)
3 (270 degrees clockwise)
"""
POSSIBLE_MOVES = [
    face + str(rotation)
    for face in ["U", "D", "F", "B", "L", "R", "R"]
    for rotation in [1, 2, 3]
]


class Cube:
    def __init__(self, flat_array) -> None:
        assert len(flat_array) == 54
        assert type(flat_array) is list

        self.faces = {
            "up": flat_array[:9].copy(),
            "down": flat_array[9:18].copy(),
            "front": flat_array[18:27].copy(),
            "back": flat_array[27:36].copy(),
            "left": flat_array[36:45].copy(),
            "right": flat_array[45:54].copy(),
        }

        self.state = flat_array

    def get_state(self):
        # return 1d list of faces
        return (
            self.faces["up"]
            + self.faces["down"]
            + self.faces["front"]
            + self.faces["back"]
            + self.faces["left"]
            + self.faces["right"]
        )

    def _rotate_face_clockwise(self, face_content):
        """Rotate a 3x3 face represented as 9-element list 90° clockwise"""
        # Map: [0,1,2,3,4,5,6,7,8] -> [6,3,0,7,4,1,8,5,2]
        rotated = [0] * 9
        rotated[0] = face_content[6]  # top-left <- bottom-left
        rotated[1] = face_content[3]  # top-center <- center-left
        rotated[2] = face_content[0]  # top-right <- top-left
        rotated[3] = face_content[7]  # center-right <- bottom-center
        rotated[4] = face_content[4]  # center <- center (unchanged)
        rotated[5] = face_content[1]  # center-right <- top-center
        rotated[6] = face_content[8]  # bottom-right <- bottom-right
        rotated[7] = face_content[5]  # bottom-center <- center-right
        rotated[8] = face_content[2]  # bottom-left <- top-right
        return rotated

    def _rotate_face(self, face_name, times):
        """Rotate a face 1, 2, or 3 times clockwise"""
        for _ in range(times):
            self.faces[face_name] = self._rotate_face_clockwise(self.faces[face_name])

    def do_rotation(self, rot):
        rotation = rot.upper()
        face = rotation[0]
        times = int(rotation[1]) if len(rotation) > 1 else 1

        if face == "U":
            self._rotate_up(times)
        elif face == "D":
            self._rotate_down(times)
        elif face == "F":
            self._rotate_front(times)
        elif face == "B":
            self._rotate_back(times)
        elif face == "L":
            self._rotate_left(times)
        elif face == "R":
            self._rotate_right(times)

    def _rotate_up(self, times):
        for _ in range(times):
            # Rotate the up face itself
            self._rotate_face("up", 1)

            # Save front edge
            temp = [
                self.faces["front"][0],
                self.faces["front"][1],
                self.faces["front"][2],
            ]

            # Front <- Right
            self.faces["front"][0] = self.faces["right"][0]
            self.faces["front"][1] = self.faces["right"][1]
            self.faces["front"][2] = self.faces["right"][2]

            # Right <- Back
            self.faces["right"][0] = self.faces["back"][0]
            self.faces["right"][1] = self.faces["back"][1]
            self.faces["right"][2] = self.faces["back"][2]

            # Back <- Left
            self.faces["back"][0] = self.faces["left"][0]
            self.faces["back"][1] = self.faces["left"][1]
            self.faces["back"][2] = self.faces["left"][2]

            # Left <- temp (original front)
            self.faces["left"][0] = temp[0]
            self.faces["left"][1] = temp[1]
            self.faces["left"][2] = temp[2]

    def _rotate_down(self, times):
        for _ in range(times):
            # Rotate the down face itself
            self._rotate_face("down", 1)

            # Save front edge
            temp = [
                self.faces["front"][6],
                self.faces["front"][7],
                self.faces["front"][8],
            ]

            # Front <- Left
            self.faces["front"][6] = self.faces["left"][6]
            self.faces["front"][7] = self.faces["left"][7]
            self.faces["front"][8] = self.faces["left"][8]

            # Left <- Back
            self.faces["left"][6] = self.faces["back"][6]
            self.faces["left"][7] = self.faces["back"][7]
            self.faces["left"][8] = self.faces["back"][8]

            # Back <- Right
            self.faces["back"][6] = self.faces["right"][6]
            self.faces["back"][7] = self.faces["right"][7]
            self.faces["back"][8] = self.faces["right"][8]

            # Right <- temp (original front)
            self.faces["right"][6] = temp[0]
            self.faces["right"][7] = temp[1]
            self.faces["right"][8] = temp[2]

    def _rotate_front(self, times):
        for _ in range(times):
            # Rotate the front face itself
            self._rotate_face("front", 1)

            # Save up edge
            temp = [self.faces["up"][6], self.faces["up"][7], self.faces["up"][8]]

            # Up <- Left
            self.faces["up"][6] = self.faces["left"][8]
            self.faces["up"][7] = self.faces["left"][5]
            self.faces["up"][8] = self.faces["left"][2]

            # Left <- Down
            self.faces["left"][2] = self.faces["down"][0]
            self.faces["left"][5] = self.faces["down"][1]
            self.faces["left"][8] = self.faces["down"][2]

            # Down <- Right
            self.faces["down"][0] = self.faces["right"][6]
            self.faces["down"][1] = self.faces["right"][3]
            self.faces["down"][2] = self.faces["right"][0]

            # Right <- temp (original up)
            self.faces["right"][0] = temp[0]
            self.faces["right"][3] = temp[1]
            self.faces["right"][6] = temp[2]

    def _rotate_back(self, times):
        for _ in range(times):
            # Rotate the back face itself
            self._rotate_face("back", 1)

            # Save up edge
            temp = [self.faces["up"][0], self.faces["up"][1], self.faces["up"][2]]

            # Up <- Right
            self.faces["up"][0] = self.faces["right"][2]
            self.faces["up"][1] = self.faces["right"][5]
            self.faces["up"][2] = self.faces["right"][8]

            # Right <- Down
            self.faces["right"][2] = self.faces["down"][8]
            self.faces["right"][5] = self.faces["down"][7]
            self.faces["right"][8] = self.faces["down"][6]

            # Down <- Left
            self.faces["down"][6] = self.faces["left"][0]
            self.faces["down"][7] = self.faces["left"][3]
            self.faces["down"][8] = self.faces["left"][6]

            # Left <- temp (original up)
            self.faces["left"][0] = temp[2]
            self.faces["left"][3] = temp[1]
            self.faces["left"][6] = temp[0]

    def _rotate_left(self, times):
        for _ in range(times):
            # Rotate the left face itself
            self._rotate_face("left", 1)

            # Save up edge
            temp = [self.faces["up"][0], self.faces["up"][3], self.faces["up"][6]]

            # Up <- Back
            self.faces["up"][0] = self.faces["back"][8]
            self.faces["up"][3] = self.faces["back"][5]
            self.faces["up"][6] = self.faces["back"][2]

            # Back <- Down
            self.faces["back"][2] = self.faces["down"][6]
            self.faces["back"][5] = self.faces["down"][3]
            self.faces["back"][8] = self.faces["down"][0]

            # Down <- Front
            self.faces["down"][0] = self.faces["front"][0]
            self.faces["down"][3] = self.faces["front"][3]
            self.faces["down"][6] = self.faces["front"][6]

            # Front <- temp (original up)
            self.faces["front"][0] = temp[0]
            self.faces["front"][3] = temp[1]
            self.faces["front"][6] = temp[2]

    def _rotate_right(self, times):
        for _ in range(times):
            # Rotate the right face itself
            self._rotate_face("right", 1)

            # Save up edge
            temp = [self.faces["up"][2], self.faces["up"][5], self.faces["up"][8]]

            # Up <- Front
            self.faces["up"][2] = self.faces["front"][2]
            self.faces["up"][5] = self.faces["front"][5]
            self.faces["up"][8] = self.faces["front"][8]

            # Front <- Down
            self.faces["front"][2] = self.faces["down"][2]
            self.faces["front"][5] = self.faces["down"][5]
            self.faces["front"][8] = self.faces["down"][8]

            # Down <- Back
            self.faces["down"][2] = self.faces["back"][6]
            self.faces["down"][5] = self.faces["back"][3]
            self.faces["down"][8] = self.faces["back"][0]

            # Back <- temp (original up)
            self.faces["back"][0] = temp[2]
            self.faces["back"][3] = temp[1]
            self.faces["back"][6] = temp[0]
